import numpy as np


# These functions may be customized to specific needs, such as accomodating to
# the expectations of Neural Networks trained outside our environment


def get_model_information():
    """Model informations for end user

    Returns
    -------
        str Information message that will appear in the Deep-Learning
        models information port.
    """

    model_information = "<html>This network is a <a href=\"https://arxiv.org/abs/1505.04597\"> \
            U-Net model variant.</a></html>"
    return model_information


# The formatting of the image into the array x is ensured by the Deep Learning Prediction module.
def preprocess_input(x):
    """Standardization of the input array x to zero mean and unit variance.
    The x array shape must be
    - (number, width, height, 1)

    Parameters
    ----------
        x : numpy array
            array to process

    Returns
    -------
        processed array
    """
    if len(x.shape) != 4:
        raise ValueError("incorrect 'x' shape, must be a 4d array with shape=(number, width, height, 1)")

    n, _, _, _ = x.shape

    if isinstance(x, np.ndarray):
        x_tmp = np.reshape(x, [x.shape[0], np.product(x.shape[1:])])
        x_mean = np.mean(x_tmp, dtype=np.float32, axis=1).reshape((n, 1, 1, 1))
        x_std = np.std(x_tmp, dtype=np.float32, axis=1).reshape((n, 1, 1, 1))
        return (x-x_mean)/np.maximum(x_std, 1e-7)
    else:
        raise TypeError("'x' must be a numpy array")


# This post-processing example casts the results to 8-bits.
def postprocess_output(y_pred):
    """This function resamples the float32 y_pred array
    which is expected to lie in the interval [0;1]
    to an 8-bits unsigned integer with values in [0;255].
    The resulting image is therefore of this type

    The y_pred array shape must be
    - (number, width, height, 1)

    Parameters
    ----------
        y_pred : array
            model prediction output

    Returns
    -------
        processed array
    """
    n, _, _, _ = y_pred.shape

    postprocess_min = np.ones((n, 1, 1, 1)) * 0.0
    postprocess_max = np.ones((n, 1, 1, 1)) * 255.0

    y_pred_norm = (y_pred*(postprocess_max - postprocess_min) + postprocess_min).astype(np.uint8)

    return y_pred_norm
