# -*- coding: utf-8 -*-

"""
moduletools contains commonly used functions to interact with
Amira/Avizo modules.
"""

from _hx_core import hx_project as _hx_project

def results_to_list(compute_module,delete_module=False):
    """
    Stores all results of a module in a list.
    
    Parameters
    ----------
    compute_module : HxCompModule
        The handle to the compute module with results.
    delete_module :  Bool
        Choice of whether to delete the module after its application.
        If False, (default) module is kept after getting result list.
        If True, module is removed from Project after getting result
        list.
    
    Returns
    -------
    out : list of HxData objects
        List of handles to data objects that are results of compute
        module.
    """
    if len(compute_module.results) == 1:
        mr = compute_module.results[0]
    else:
        mr = []
        for i,r in enumerate(compute_module.results):
            mr.append(compute_module.results[i])
    if delete_module is True:
        _hx_project(compute_module)
    return mr

def get_cell_contents(tablelike_object,row,column,table=0):
    """
    Retrieves contents of cell i,j in a table through
    HxSpreadSheetInterface.
    
    Parameters
    ----------
    tablelike_object : HxLabelAnalysis, HxSpreadSheet, HxImageAnalysis
        The handle to the table-like object.
    row : Int
        Row number from which to retrieve value (starting at 0).
    column : Int
        Column number from which to retrieve value (starting at 0).
    table : Int
        Table (i.e. tab) from which to retrieve value (starting at 0).
        Defaults to the first (or only) tab. Useful for results of
        spatial graphs, where there are 3+ tabs.
    
    Returns
    -------
    out : See note
        Contents of the specified cell.
        Note: returned valued has same type of column the item is
        received from.
    """
    ssi = tablelike_object.all_interfaces.HxSpreadSheetInterface
    return ssi.tables[table].items[row,column]

def labelanalysis_to_dataframe(tablelike_object):
    """
    Converts a tablelike object (HxSpreadSheet, HxLabelAnalysis,
    or anything with .all_interfaces.HxSpreadSheetInterface) to a
    Pandas data frame with the column names intact.
    
    Parameters
    ----------
    tablelike_object : HxLabelAnalysis, HxSpreadSheet, HxImageAnalysis
        The handle to the table-like object.
    
    Returns
    -------
    out : Dataframe
        Entire contents of the table-like object.
    """
    import numpy as np
    import pandas as pd
    # Grab interface to data in spreadsheet
    interface = tablelike_object.all_interfaces.HxSpreadSheetInterface
    # Count number of columns/rows
    nrows = len(interface.tables[0].rows)
    ncols = len(interface.tables[0].columns)
    # Create a list of column names in spreadsheet
    sscl = []
    for i in range(0,ncols):
        sscl.append(interface.tables[0].columns[i].name)
    # Get data from spreadsheet into an array
    ssnda = np.empty([nrows,ncols])
    for i in range(0,nrows):
        for j in range(0,ncols):
            try:
                ssnda[i,j] = interface.tables[0].items[i,j]
            except ValueError:
                ssnda[i,j] = 0
    # Pass array to data frame with correct column names
    ssdf_b = pd.DataFrame(data=ssnda,index=range(0,nrows),columns=sscl)
    return ssdf_b
    # TODO
    # Delete left row of table that is full of zeros - 
    #    only happens sometimes
    # Explore bug where column names can be wiped out

# TODO
#