# -*- coding: utf-8 -*-

"""
tcltunnel contains wrappers to easily send TCL commands through
_tcl_interp from the Python console.

The functions that are currently implemented include support for:
    -   Setting the message and state of the progress bar
    -   Setting certain module port settings that are not currently
        supported in the Python API
    -   Saving data objects to disk as .am format
    -   Interacting with the time slider of the Time Series Control
        module

Documentation of the wrapped TCL commands is provided where applicable,
and can also be found in the Avizo/Amira User's Guide.
"""

from _hx_core import _tcl_interp

# Functions for working with the progress bar
# -------------------------------------------

def start_working(message="Computing…"):
    """
    Sets the progress bar status to working with a custom message.
    
    Parameters
    ----------
    message : String
        The message to display in the progress bar while working.
    
    TCL Documentation
    -----------------
    workArea startWorking [<message>]
        Activates the stop button. After calling this command, the
        Avizo/Amira stop button becomes active. In your script, you can
        check if the stop button was hit by calling
        workArea wasInterrupted. When the stop button is active, you
        can't interact with any other widget unless you call
        workArea stopWorking in your script. Therefore, you must not
        enter this command directly in the console window, but you
        should only use it in a script file or in a Tcl procedure.
    """
    tcl_string = "workArea startWorking " + "{" + str(message) + "}"
    _tcl_interp(tcl_string)

def stop_working():
    """
    Sets the progress bar status to ready and clears the stop button.
    
    TCL Documentation
    -----------------
    workArea stopWorking
        Deactivates the stop button. Call this command when the compute
        task started with workArea startWorking is done or if the user
        pressed the stop button. This command also restores the
        progress info text which was shown before calling startWorking.
    """
    tcl_string = "workArea stopWorking"
    _tcl_interp(tcl_string)

def set_progress_info(text="0/100"):
    """
    Sets the info message in the progress bar.
    
    Parameters
    ----------
    text : String
        The text to display after the start_working() message in the
        progress bar. Typically used to indicate level of computation
        completion (percentage, iteration number, etc.).
    
    TCL Documentation
    -----------------
    workArea setProgressInfo <text>
        Sets an info text to be displayed in the progress bar. The text
        can be used to describe the status during some computation.
    """
    tcl_string = "workArea setProgressInfo " + "{" + str(text) + "}"
    _tcl_interp(tcl_string)

def set_progress_value(value=0.0):
    """
    Sets the value of the progress bar, which translates to how far
    across the screen the red progress indicator spans.
    
    Parameters
    ----------
    value : Float
        A number between 0.0 and 1.0 that defines the length of the red
        computation indicator that increases across the progress bar.
    
    TCL Documentation
    -----------------
    workArea setProgressValue <value>
        Sets the value of the progress bar. The argument must be a
        floating point number between 0 and 1. For example, a value of
        0.8 indicates that 80% of the current task has been done.
    """
    
    # TODO
    # Check if value parameter is outside [0.0,1.0] for set_progress_value()
    
    tcl_string = "workArea setProgressValue " + str(value)
    _tcl_interp(tcl_string)

def clear_progress_info():
    """
    Clears the progress bar by setting its value to 0.0 and the status
    to "Ready" (the default value used by Avizo/Amira when no
    computation is being performed.
    """
    _tcl_interp("workArea setProgressValue 0")
    _tcl_interp("workArea setProgressInfo Ready")


# Functions for working with port settings
# ----------------------------------------

def set_measures_group(label_analysis_module,measures="basic",custom=False):
    """
    Sets the measures group for a Label Analysis module.

    Parameters
    ----------
    label_analysis_module : HxLabelAnalysis
        The handle to the Label Analysis module.
    measures : String
        The measures group to set in the Measures port.
        The standard measures groups are valid for all installations:
            basic (default)
            basic2D
            Standard Shape Analysis
            Weighted Shape Analysis
        If custom is False, this port needs the setState string that
        defines the custom measures group.
    custom :  Bool
        Whether or not the measures group is pre-defined.
        False (default) if the measures group not already defined.
        True if measures group needs to be defined.
    
    TCL Documentation
    -----------------
    In the Avizo/Amira User's Guide, see the "Scripting tips" section
    at the bottom of the page titled "More aboout label measures"
    (accessible via the "Search Help:" bar in the top-right of the
    User's Guide.
    """
    if custom is False:
        tcl_string = "{" + label_analysis_module.name + "} measures setValue " + "{" + measures + "}"
    else:
        tcl_string = "{" + label_analysis_module.name + "} measures setState " + "{" + measures + "}"
    _tcl_interp(tcl_string)

def create_label_measure(name,formula):
    """
    Checks if a label measure exists and, if not, creates it.
    
    See the TCL console for errors if an invalid formula is passed.
    
    Parameters
    ----------
    name : String
        Name of the label measure you wish to create.
    formula : String
        Defines the formula for the measure.
    
    TCL Documentation
    -----------------
    In the Avizo/Amira User's Guide, see the "Scripting tips" section
    at the bottom of the page titled "More aboout label measures"
    (accessible via the "Search Help:" bar in the top-right of the
    User's Guide.
    """
    tcl_string = 'lsearch [labelMeasure list] \"{}\"'.format(name)
    currently_exists = _tcl_interp(tcl_string)
    if currently_exists == '-1':
        tcl_string = 'labelMeasure create \"{}\" nounit \"{}\"'.format(name,formula)
        _tcl_interp(tcl_string)


# Functions for working with data objects
# ---------------------------------------

def save_data_object(data_object,directory):
    """
    Saves a data object in the Amira mesh binary format (.am).
    
    Parameters
    ----------
    data_object : HxData
        Handle for the dataset to be saved.
    directory : String
        Location to save the dataset.
    
    TCL Documentation
    -----------------
    exportData [format] [filename]
        Export the data object. A format and a file name must be
        specified. The format should be the name of a format as it is
        displayed in the file dialog's file type menu, e.g.,
        "Avizo ascii" or "HxSurface binary". When an object is exported
        its name is replaced by a possibly modified version of the
        filename. The command returns the actual name of the object
        after it has been exported.
    """
    
    # TODO
    # Add support for variable file format strings
    
    tcl_string = "{" + data_object.name + "} exportData \"Amira binary\" {" + directory + "/" + data_object.name + "}"
    _tcl_interp(tcl_string)

    
# Functions for working with time series control
# ----------------------------------------------

def get_time(tsc_object):
    """
    Retrieves the index of the currently selected time step.
    
    Parameters
    ----------
    tsc_object : HxDynamicFileSeriesCtrl
        Handle for the time series control module.
    
    Returns
    -------
    out : Int
        Currently selected time step.
    
    TCL Documentation
    -----------------
    getValue
        Returns the current time value.
    """
    tcl_string = "{"+tsc_object.name+"}" + " time getValue "
    return(int(_tcl_interp(tcl_string)))

def get_max_time(tsc_object):
    """
    Retrieves the maximum loaded time step using TCL's lindex to access
    second value of the two values (min and max) returned by this
    specific command (see TCL documentation below).
    
    Parameters
    ----------
    tsc_object : HxDynamicFileSeriesCtrl
        Handle for the time series control module.
    
    Returns
    -------
    out : Int
        Maximum time step.
    
    TCL Documentation
    -----------------
    getMinMax
        Returns two numbers indicating the full range of the time port.
    """
    tcl_string = "lindex [" + "{"+tsc_object.name+"}" + " time getMinMax] 1 "
    return(int(_tcl_interp(tcl_string)))

def set_time(tsc_object,current_time):
    """
    Sets the current time to the given value.
    
    Parameters
    ----------
    tsc_object : HxDynamicFileSeriesCtrl
        Handle for the time series control module.
    current_time : Int
        The desired time step number to step to.
    
    TCL Documentation
    -----------------
    setValue <value>
        Sets the current time value.
    """
    tcl_string = "{"+tsc_object.name+"}" + " time setValue " + str(current_time)
    _tcl_interp(tcl_string)
    tcl_string = "{"+tsc_object.name+"}" + " fire"
    _tcl_interp(tcl_string)

def get_current_file_basename(tsc_object):
    """
    Queries the base filename of currently selected time step data
    object (i.e. what the data file is named on disk).
    
    Parameters
    ----------
    tsc_object : HxDynamicFileSeriesCtrl
        Handle for the time series control module.
    
    Returns
    -------
    out : String
        Filename of the currently selected time step.
    """
    tcl_string = "{"+tsc_object.name+"}" + " getCurrentFileBasename"
    return(_tcl_interp(tcl_string))

def get_current_time_step_name(tsc_object):
    """
    Queries the label of the currently selected time step data object.
    
    Parameters
    ----------
    tsc_object : HxDynamicFileSeriesCtrl
        Handle for the time series control module.
    
    Returns
    -------
    out : String
        Label of the currently selected time step.
    """
    tcl_string = "{"+tsc_object.name+"}" + " getResult"
    return(_tcl_interp(tcl_string))    

def get_current_multichannel(tsc_object):
    """
    Queries the label of the currently selected multi-channel field.
    
    Parameters
    ----------
    tsc_object : HxDynamicFileSeriesCtrl
        Handle for the time series control module.
    
    Returns
    -------
    out : String
        Label of the currently selected multi-channel field.
    """
    tcl_string = "lindex [lindex [" + "{"+tsc_object.name+"}" + " downStreamConnections] 0] 0"
    return(_tcl_interp(tcl_string))

