"""Preferences class
   Provide some methods to add preferences and section to a file then load it ( using TCL interpreter.)
   Provides method to read in orderedDict the current application preferences
   Provides method to write in orderedDict and apply to application preferences
Example
-------
>>> from Preferences import *
>>> pref = Preferences.Preferences()
>>> pref.add_section()
>>> pref.add_item("test item", "4.0")
>>> pref.add_item("QA", True)
>>> pref.add_section("customSettings")
>>> pred.add_item("custom", "99")
>>> pref.commit_changes()
>>> pref.restore_default()
>>> pref_dict=pref.get_values()
>>> print(pref_dict['localDir'])
>>> print(pref_dict['Units DU Length'])
>>> pref_dict['Units DU Length'] = 'cm'
>>> pref.commit_changes()
>>> with Preferences() as prefs:
>>>   prefs["test item"] = '4.0' 
>>>   prefs["QA"] = 'true'
>>>   print(prefs['localDir'])
>>>   print(prefs['Units DU Length'])
>>>   prefs['Units DU Length'] = 'cm'
"""
import os
from _hx_core import HxPaths
from collections import OrderedDict

class Preferences:

    @staticmethod    
    def save_preferences(filename):
        from _hx_core import _tcl_interp
        _tcl_interp("thePreferences save "+str(os.path.join(HxPaths().home_dir,filename)).replace("\\","\\\\"))
        
    @staticmethod    
    def load_preferences(filename):
        from _hx_core import _tcl_interp
        _tcl_interp("thePreferences load "+str(os.path.join(HxPaths().home_dir,filename)).replace("\\","\\\\"))    

    @staticmethod
    def restore_default():
        from _hx_core import _tcl_interp
        _tcl_interp("thePreferences reset")

    def __init__(self, filename="preferences.pref"):
        self._preferences = OrderedDict()
        self._filename = str(filename)
        self._load_items()
        
    def __enter__(self):
        return self.get_values()

    def __exit__(self, exception_type, exception_value, traceback):
        self._preferences.commit_changes()        

    def add_item(self, item, value):
        """
        add_item: Add one preference, the path to the preference is a blank separated string 
                  the preference change must be commited by mean of a  commit_changes() method call
        """    
        self._preferences.update(OrderedDict.fromkeys([item], value))

    def del_item(self, item):
        """
        del_item: Delete one preference, the path to the preference is a blank separated string 
                  the preference change must be commited by mean of a  commit_changes() method call
        """      
        self._preferences.pop(item)

    def reset_preferences(self):
        """
        reset_preferences: reset all previous add_item, del_item and add_section 
        """ 
        self._preferences.clear()

    def add_section(self, section="preferences"):
        """
        add_section: Add a section in the preferences, known section names are 'preferences' and 'customSettings'
                     the preference change must be commited by mean of a  commit_changes() method call
        """     
        section = str("[")+section+str("]")
        self.add_item(section, section) 
        
    def _read_file(self, filename):
        with open(os.path.join(HxPaths().home_dir,filename), "r") as file:
            for line in file:
                tokens = line.split("=")
                if len(tokens) == 1:
                    # handles section case
                    key=tokens[0].replace("\n", "")
                    #do not use add_section to avoid double []
                    self.add_item(key,key)
                elif len(tokens) > 1:
                    # handles cases in which the value is containing =
                    key=tokens[0].replace("\\", " ")
                    val='='.join(tokens[1:]).replace("\n", "")
                    self.add_item(key, val)
            file.close()
        
    def _load_items(self):
        """
        _load_items: Load ordered dictionary with current application preferences
        """
        Preferences.save_preferences("to_read_pref.pref")
        self._read_file("to_read_pref.pref")
        os.remove(os.path.join(HxPaths().home_dir,"to_read_pref.pref"))
        
    def get_values(self):
        """
        get_values: returns OrderedDict of current preferences (to read)
        """
        return self._preferences
        
    def _write_file(self, filename):
        with open(os.path.join(HxPaths().home_dir,filename), "w") as file:
            for item, val in self._preferences.items():
                string_to_write = str(item)+"\n" if item == val else str(item.replace(" ", "\\"))+"="+str(val)+"\n"
                file.write(string_to_write)
            file.close()
            
    def _test_write_read(self):
        """
        _test_write_read: this is a test checking if de-serialization from file and serialization to file provide the same file
        """     
        from filecmp import cmp
        Preferences.save_preferences("to_read_pref.pref")
        self._read_file("to_read_pref.pref")
        self._write_file("test_write_pref.pref")
        retVal = False
        if cmp(os.path.join(HxPaths().home_dir,"to_read_pref.pref"), os.path.join(HxPaths().home_dir,"test_write_pref.pref") ):
            retVal=True
        os.remove(os.path.join(HxPaths().home_dir,"to_read_pref.pref"))
        os.remove(os.path.join(HxPaths().home_dir,"test_write_pref.pref"))
        return retVal
        
    def _save_preferences(self):
        if self._filename != "":
            self._write_file(self._filename)
        else:
            return None
            
    def _remove_preferences(self):
        os.remove(os.path.join(HxPaths().home_dir,self._filename))

    def commit_changes(self):
        """
        commit_changes: Apply the preferences set by mean of  previous add_section, del_item, add_item calls 
        """ 
        from _hx_core import _tcl_interp
        self._save_preferences()
        _tcl_interp("thePreferences load "+str(os.path.join(HxPaths().home_dir,self._filename)).replace("\\","\\\\"))
        self._remove_preferences()

    @staticmethod
    def _set_units_management_enabled(enabled=True):
        """Enable or disable the units management preference

        Parameters
        ----------
        enabled: boolean
            Whether the units should be enabled, True by default

        Examples
        --------

        >>> from QAHelpers import Preferences
        >>> Preferences._set_units_management_enabled(False)
        """
        pref = Preferences()
        pref.add_section("customSettings")

        # The following int values correspond to the appropriate enum values defined in 'hxunits\internal\QxUnitsTclManager.cpp':
        #  - UnitsNone = 0x1
        #  - UnitsSpatialInfoOnly = 0x2
        #  - (UnitsFull = 0x4)
        if enabled:
            pref.add_item("Units UnitsManagementMode", "2")
        else:
            pref.add_item("Units UnitsManagementMode", "1")

        pref.load_preferences()

    @staticmethod
    def enable_units_management():
        """Enable the units management preference

        Examples
        --------

        >>> from QAHelpers import Preferences
        >>> Preferences.enable_units_management()
        """
        Preferences._set_units_management_enabled(True)

    @staticmethod
    def disable_units_management():
        """Disable the units management preference

        Examples
        --------

        >>> from QAHelpers import Preferences
        >>> Preferences.disable_units_management()
        """
        Preferences._set_units_management_enabled(False)
