"""UnitOfMeasureFixer class
   Provides helper method to "inherit" units of measure from an object  
   temporary without using and changing the current um preferences 

Example
-------
>>> from UnitOfMeasuresFixer import *
>>> surf_cut_obj = hx_project.get('Surface Cross Section')
>>> surf=hx_project.get('chocolate-bar.surf')
>>> surf_cut_obj.ports.data.connect(surf)
>>> export_port = surf_cut_obj.ports.export
>>> create_surf_btn = export_port.buttons[0]
>>> with UnitOfMeasuresFixer() as udm:
>>>     create_surf_btn.hit = True
>>>     udm.set_units_from_object(surf)
>>>     surf_cut_obj.fire()

"""

import _hx_core
from Preferences import *
import ParamBundleWrapper

class UnitOfMeasuresFixer():
    def __init__(self):
        self._bck_pref_filename = 'bck_preferences.pref'
        
    def __enter__(self): 
        self.backup_prefs = Preferences()
        self._prefs = Preferences()
        return self
  
    def __exit__(self, exception_type, exception_value, traceback):
        # restore preferences but for check enableDoNotShowAgainButtonForUnits 
        pref_dict = self._prefs.get_values()
        try:
            enableDoNotShowAgainButtonForUnits = bool(pref_dict['enableDoNotShowAgainButtonForUnits']=='true')
            if enableDoNotShowAgainButtonForUnits:
                self.backup_prefs.add_item("enableDoNotShowAgainButtonForUnits", "true")
                self.backup_prefs.add_item("Units ShowUnitsDialogWhenLoadingData", "0")            
        except KeyError:
            pass

        self.backup_prefs.commit_changes()
        
    def set_units_from_object(self, obj):
        """
        set_units_from_object: set temporary current unit of measures of the input object

        Parameters
        ----------
        obj: object having units of measure to use
        """    
        units_dict = { }
        parameters = {'Units' : units_dict}
        ParamBundleWrapper.get_parameter_bundle(obj, parameters)
        if parameters['Units']['Coordinates'] is not None:
            self._prefs.add_section("customSettings")
            self._prefs.add_item("Units UnitsManagementMode", "2")
            self._prefs.add_item("Units AutomaticallyDetermineWU", "1")
            self._prefs.add_item("Units DefaultCoordinatesUnitAtLoading", str(parameters['Units']['Coordinates']))
            self._prefs.commit_changes()
        else:
            print("Coordinates not found on origin object")